extern crate native_windows_derive as nwd;
extern crate native_windows_gui as nwg;

use nwd::NwgUi;
use nwg::NativeUi;
use serde::Deserialize;
use std::sync::{Arc, Mutex};
use std::thread;
use std::time::Duration;
use std::{cell::RefCell, collections::HashMap};

use wmi::{COMLibrary, FilterValue, WMIConnection};

const PROCESS_NAME: &str = "Notepad3.exe";
// const PROCESS_NAME: &str = "rtcRemoteDesktop.exe";
// const PROCESS_PATH_PREFIX: &str = r"C:\Program Files (x86)\Seewo\SeewoService";
const PROCESS_PATH_INCLUDE: &str = "SeewoService";
const CHECK_INTERVAL: Duration = Duration::from_secs(1);

const WINDOW_WIDTH: i32 = 130;
const WINDOW_HEIGHT: i32 = 20;

#[derive(Default, NwgUi)]
pub struct BasicApp {
    #[nwg_control(size: (WINDOW_WIDTH, WINDOW_HEIGHT), position: (895, -18), title: "", flags: "VISIBLE")]
    #[nwg_events( OnInit: [BasicApp::setup], OnWindowClose: [BasicApp::say_goodbye] )]
    window: nwg::Window,

    #[nwg_control]
    #[nwg_events(OnNotice: [BasicApp::update_state])]
    update_state: nwg::Notice,

    state_receiver: RefCell<Option<std::sync::mpsc::Receiver<bool>>>,
}

impl BasicApp {
    fn setup(&self) {
        /*{  
            let screen_width = {
                let width: Arc<Mutex<i32>> = Arc::new(Mutex::new(1920));
                let width_ref = width.clone();
                let handle = thread::spawn(move || {
                    let com_con = COMLibrary::new().unwrap();
                    let wmi_con = WMIConnection::new(com_con).unwrap();
                    let res: Vec<VideoController> =
                        wmi_con.query().expect("Failed to get screen resolution.");
                    let mut width_locked = width_ref.lock().unwrap();
                    *width_locked = res.first().unwrap().current_horizontal_resolution;
                });
                handle.join().unwrap();
                let width_result = width.lock().unwrap().to_owned();
                width_result
            };
            // self.window.set_position((screen_width + WINDOW_WIDTH) / 2, 5);
            println!("{:?}", ((screen_width + WINDOW_WIDTH) / 2, 5));
            println!("{:?}", self.window.size())
             
        }*/
        self.window.set_visible(false);

        let (sender, receiver) = std::sync::mpsc::channel();

        // Creates a sender to trigger the `OnNotice` event
        let notice_sender = self.update_state.sender();

        thread::spawn(move || {
                let com_con = COMLibrary::new().unwrap();
                let wmi_con = WMIConnection::new(com_con).unwrap();
                let new_process_filters = {
                    let mut filters = HashMap::<String, FilterValue>::new();
                    filters.insert(
                        "TargetInstance".to_owned(),
                        FilterValue::is_a::<Process>().unwrap(),
                    );
                    filters.insert(
                        "TargetInstance.Name".to_owned(),
                        wmi::FilterValue::Str(PROCESS_NAME),
                    );
                    filters
                };

                let query_process_filters = {
                    let mut filters = HashMap::<String, FilterValue>::new();
                    filters.insert("Name".to_owned(), wmi::FilterValue::Str(PROCESS_NAME));
                    filters
                };

                let iterator = wmi_con
                    .filtered_notification::<NewProcessEvent>(
                        &new_process_filters,
                        Some(CHECK_INTERVAL),
                    )
                    .unwrap();

                for result in iterator {
                    let process = result.unwrap().target_instance;
                    // println!("PID:        {}", process.process_id);

                    // println!("Name:       {}", process.name);
                    // println!("Executable: {:.?}", process.executable_path);
                    match check_process(&process) {
                        true => {
                            println!("start");
                            sender.send(true).unwrap();
                            notice_sender.notice();
                            loop {
                                match query(&wmi_con, &query_process_filters) {
                                    false => {
                                        println!("stop");
                                        sender.send(false).unwrap();
                                        notice_sender.notice();
                                        break;
                                    }
                                    true => std::thread::sleep(CHECK_INTERVAL),
                                };
                            }
                        }
                        false => {}
                    }
                }
            }/*{
                 let mut counter = false;

                 loop {
                   counter = !counter;
                   sender.send(counter).unwrap();

                   // Calling the notice function will trigger the OnNotice event on the gui thread
                   notice_sender.notice();

                   thread::sleep(::std::time::Duration::from_millis(5000));
                 }
               }*/
            );

        *self.state_receiver.borrow_mut() = Some(receiver);
    }

    fn update_state(&self) {
        let mut receiver_ref = self.state_receiver.borrow_mut();
        let receiver = receiver_ref.as_mut().unwrap();
        while let Ok(data) = receiver.try_recv() {
            match data {
                true => {
                    self.window.set_visible(true);
                }
                false => {
                    self.window.set_visible(false);
                }
            }
        }
    }

    fn say_goodbye(&self) {
        nwg::stop_thread_dispatch();
    }
}

#[derive(Deserialize, Debug)]
#[serde(rename = "__InstanceCreationEvent")]
#[serde(rename_all = "PascalCase")]
struct NewProcessEvent {
    target_instance: Process,
}

#[derive(Deserialize, Debug)]
#[serde(rename = "Win32_Process")]
#[serde(rename_all = "PascalCase")]
struct Process {
    // process_id: u32,
    name: String,
    executable_path: Option<String>,
}

#[derive(Deserialize, Debug)]
#[serde(rename = "Win32_VideoController")]
#[serde(rename_all = "PascalCase")]
struct VideoController {
    current_horizontal_resolution: i32,
    // current_vertical_resolution: i32,
}

fn query(wmi_con: &WMIConnection, filters: &HashMap<String, FilterValue>) -> bool {
    let res: Result<Vec<Process>, wmi::WMIError> = wmi_con.filtered_query(filters);
    match res {
        Ok(v) => match v.last() {
            Some(p) => check_process(p),
            None => false,
        },
        Err(_) => false,
    }
}

fn check_process(p: &Process) -> bool {
    p.name == PROCESS_NAME.to_owned() && {
        match &p.executable_path {
            Some(path) => path.contains(PROCESS_PATH_INCLUDE),
            None => false,
        }
    }
}

fn main() {
    nwg::init().expect("Failed to init Native Windows GUI");
    nwg::Font::set_global_family("Segoe UI").expect("Failed to set default font");

    let app = BasicApp::build_ui(Default::default()).expect("Failed to build UI");

    // println!("{:?}", app.window.position());

    nwg::dispatch_thread_events();
}
